"""Repository for Assignment entities (TinyDB-backed)."""

from __future__ import annotations

import os
from datetime import date, datetime
from typing import Any, Dict, List, Optional

from tinydb import TinyDB, Query


def _iso(val: Any) -> Any:
    """Serialize date/datetime to ISO string; pass through other types."""
    if isinstance(val, (date, datetime)):
        return val.isoformat()
    return val


def _clamp_int(x: Any, lo: int = 0, hi: int = 100) -> int:
    try:
        n = int(x)
    except Exception:
        return lo
    return max(lo, min(hi, n))


class AssignmentRepository:
    """
    TinyDB repository for task assignments.
    Stores data in data/assignments.json, table 'assignments'.
    Serializes date fields and clamps percent_complete to 0..100.
    """

    def __init__(self, db_path: str = "data/assignments.json") -> None:
        os.makedirs(os.path.dirname(db_path), exist_ok=True)
        self.db = TinyDB(db_path)
        self.table = self.db.table("assignments")

    # ---------- helpers ----------
    def _next_id(self) -> int:
        rows = self.table.all()
        if not rows:
            return 1
        return max((int(r.get("id") or 0) for r in rows), default=0) + 1

    # ---------- CRUD ----------
    def create_assignment(self, assignment: Any) -> int:
        """
        Insert an assignment. Accepts dict or Pydantic model with .dict().
        Serializes 'assigned_on' and 'actual_completion_date' to ISO.
        """
        data: Dict[str, Any] = assignment.dict() if hasattr(assignment, "dict") else dict(assignment)

        # Serialize dates
        for f in ("assigned_on", "actual_completion_date"):
            if f in data:
                data[f] = _iso(data[f])

        # Clamp percent_complete
        if "percent_complete" in data:
            data["percent_complete"] = _clamp_int(data["percent_complete"])

        if not data.get("id"):
            data["id"] = self._next_id()

        self.table.insert(data)
        return int(data["id"])

    def update_assignment(self, assignment_id: int, **fields: Any) -> bool:
        """Update fields; serialize dates and clamp percent_complete."""
        for f in ("assigned_on", "actual_completion_date"):
            if f in fields:
                fields[f] = _iso(fields[f])
        if "percent_complete" in fields:
            fields["percent_complete"] = _clamp_int(fields["percent_complete"])

        q = Query()
        return bool(self.table.update(fields, q.id == assignment_id))

    def delete_assignment(self, assignment_id: int) -> bool:
        q = Query()
        return bool(self.table.remove(q.id == assignment_id))

    # ---------- queries ----------
    def get_by_id(self, assignment_id: int) -> Optional[Dict[str, Any]]:
        q = Query()
        res = self.table.get(q.id == assignment_id)
        return dict(res) if res else None

    def list_all(self) -> List[Dict[str, Any]]:
        return [dict(r) for r in self.table.all()]

    def by_task(self, task_id: int) -> List[Dict[str, Any]]:
        q = Query()
        return [dict(r) for r in self.table.search(q.task_id == task_id)]

    def by_employee(self, employee_id: int) -> List[Dict[str, Any]]:
        q = Query()
        return [dict(r) for r in self.table.search(q.employee_id == employee_id)]
